// Buffered Graphics for C
// Allows for the use of graphics buffers to perform basic operations.

#ifndef __BUF
#define __BUF

// Includes ---------------------------------------------------------------

#include <tigcclib.h>

// Constants --------------------------------------------------------------

#ifdef ti89
#define LCDWidth 160
#define LCDHeight 100
#endif
#ifdef ti92plus
#define LCDWidth 240
#define LCDHeight 128
#endif

#define LCDByteWidth (240 / 8)
#define LCDByteSize LCD_SIZE
#define LCDRealByteWidth (LCDWidth / 8)
#define LCDRealByteSize (LCDByteWidth * LCDHeight)

#define BitOn 1
#define BitOff 0

// Types ------------------------------------------------------------------

// Graphics buffer for background operations
typedef struct
{
	int PlaneCount;                       // Use InitBuffer to set this
	LCD_BUFFER Planes [0];                // Plane 0 is dark gray plane
} GRAPH_BUFFER;
typedef GRAPH_BUFFER TGraphBuffer;
typedef TGraphBuffer* PGraphBuffer;
typedef LCD_BUFFER TLCDBuffer;
typedef TLCDBuffer* PLCDBuffer;

// Color definitions
typedef unsigned char TColor;           // Grayscale color using 0x000000 (Black) thru 0xFFFFFF (White)

#define clGray0   0x00
#define clGray17  0x2A
#define clGray25  0x3F
#define clGray33  0x55
#define clGray50  0x7F
#define clGray67  0xAA
#define clGray75  0xBF
#define clGray83  0xD5
#define clGray100 0xFF

#define clWhite       clGray0
#define clLightSilver clGray33
#define clSilver      clGray50
#define clGray        clGray67
#define clBlack       clGray100

// Functions --------------------------------------------------------------

void GraphInit(int PlaneCount);         // Initialize gray mode and plane array
void GraphDone(void);                   // Free plane array and turn gray off
PGraphBuffer CreateBuf(int PlaneCount);
                                        // Initialize a buffer
void ClrBuf(PGraphBuffer Buffer);       // Clear the contents of a buffer
void FlushBuf(PGraphBuffer Buffer);     // Draw the contents of the buffer on the LCD
void ActivateBuf(PGraphBuffer Buffer, int Plane);
                                        // Activate a plane in the buffer for standard drawing
                                        // Do not use this unless implementing the function to draw directly into the buffer is impossible or inefficient.

void DrawHLineBuf(PGraphBuffer Buffer, int X1, int X2, int Y, TColor Color);
                                        // Draw a horizontal line from X1 to X2
void DrawVLineBuf(PGraphBuffer Buffer, int X, int Y1, int Y2, TColor Color);
                                        // Draw a vertical line from Y1 to Y2
void DrawFullHLineBuf(PGraphBuffer Buffer, int Y, TColor Color);
                                        // Draw a complete horizontal line
void DrawFullHLinesBuf(PGraphBuffer Buffer, int Y1, int Y2, TColor Color);
                                        // Draw complete horizontal lines from Y1 to Y2, thereby filling the area between Y1 and Y2
void DrawRectBuf(PGraphBuffer Buffer, int X1, int X2, int Y1, int Y2, TColor Color);
                                        // Draw a rectangle

unsigned char GetDitherMask(int Plane, TColor Color, int Y);

// Macros -----------------------------------------------------------------

#define FreeBuf(Buffer) free (Buffer)
#define DeactivateBuf PortRestore

#define PlaneIsBlack(Plane, Color) ((Color) & (1 << (7 - (Plane))))
#define GetDitherVal(Color) (((unsigned char) ((Color) >> (7 - CurGrayPlanes))) & 0x07)
#define IsMajorDithered(Val) ((Val) == 3 || (Val) == 4)
#define IsMinorDithered(Val) ((Val) == 1 || (Val) == 3 || (Val) == 4 || (Val) == 6)
#define GetBufPos(X, Y) (((Y) * LCDByteWidth) + ((X) >> 3))
#define GetByteMask(Start, End) ((1 << ((End) + 1)) - (1 << (Start)))

// Variables --------------------------------------------------------------

extern int CurGrayPlanes;               // Specify grayscale mode
                                        // 1: B/W; 2: Gray-4
                                        // Must be set manually
extern PLCDBuffer *GrayPlanes;          // Array of planes, or LCD memory in B/W mode
                                        // 0 is dark gray plane

#endif
